import type { NextApiRequest, NextApiResponse } from "next";
import { getServerSession } from "next-auth";
import { authOptions } from "@/lib/authOptions";
import { prisma } from "@/lib/prisma";
import { z } from "zod";

const CreateBody = z.object({
  name: z.string().trim().min(1).max(64),
  locationId: z.string().cuid(),
});

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  if (req.method === "GET") {
    const session = await getServerSession(req, res, authOptions);
    const isAdmin = (session?.user as any)?.role === "ADMIN";
    const showHidden = req.query.showHidden === "true" && isAdmin;
    const locationId = (req.query.locationId as string) || undefined;
    const search = req.query.search as string;

    const whereClause: any = {};
    if (locationId) whereClause.locationId = locationId;
    if (!showHidden) whereClause.isVisible = true;

    // Search filter
    if (search) {
      whereClause.name = { contains: search, mode: "insensitive" };
    }

    const items = await prisma.element.findMany({
      where: whereClause,
      orderBy: [{ locationId: "asc" }, { name: "asc" }],
    });
    return res.status(200).json({ ok: true, items });
  }

  if (req.method === "POST") {
    const session = await getServerSession(req, res, authOptions);
    if ((session?.user as any)?.role !== "ADMIN")
      return res.status(403).json({ ok: false, error: "Forbidden" });

    const p = CreateBody.safeParse(req.body);
    if (!p.success)
      return res.status(400).json({ ok: false, error: "Invalid body" });

    const loc = await prisma.location.findUnique({
      where: { id: p.data.locationId },
    });
    if (!loc)
      return res.status(400).json({ ok: false, error: "Invalid locationId" });

    const item = await prisma.element.create({
      data: { name: p.data.name, locationId: p.data.locationId },
    });
    return res.status(201).json({ ok: true, item });
  }

  return res.status(405).json({ ok: false, error: "Method not allowed" });
}
